//*************************************************************************************************
//
//	Description:
//		ground_snowcover.fx - Snowcover object shader for The Race. A specialisation of ground.fx with
//													an extra layer which blends in according to the orientation of faces.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     02/07/2007  0.1           Created
//		TNettleship			02/13/2007	0.2						Fixed normal map inconsistency between max & engine
//		TNettleship			04/18/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias.
//		TNettleship			10/23/2007	2.04					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

#endif


//
// Textures
//

float snowThreshold														// Threshold below which snow stops (vector component, not an angle)
<
	string UIName = "Snow threshold";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.5f;

float snowFade																// Vector amount during which snow fades in
<
	string UIName = "Snow fade";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.1f;

#ifdef _3DSMAX_
texture broadDiffuseTexture : DiffuseMap			// Broad diffuse colour in RGB
#else
texture broadDiffuseTexture : TEXTURE					// Broad diffuse colour in RGB
#endif
<
	string UIName = "Broad Diffuse";
	bool appEdit = true;
>;

texture middleDiffuseTexture : TEXTURE				// Middle diffuse colour in RGB
<
	string UIName = "Middle Diffuse";
	bool appEdit = true;
>;

texture detailDiffuseTexture : TEXTURE				// Detail diffuse colour in RGB
<
	string UIName = "Detail Diffuse";
	bool appEdit = true;
>;

float detailUScale														// Detail layer U scale
<
	string UIName = "Detail Layer U Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;

float detailVScale														// Detail layer V scale
<
	string UIName = "Detail Layer V Scale";
	float UIMin = -128.0f;
	float UIMax = 128.0f;
	bool appEdit = true;
> = 1.0f;

float detailUVRotation												// Detail layer UV rotation (degrees)
<
	string UIName = "Detail Layer UV Rotation";
	float UIMin = 0.0f;
	float UIMax = 360.0f;
	float UIStep = 1.0f;
	bool appEdit = true;
> = 0.0f;

bool additiveDetail														// Additive detail?
<
	string UIName = "Additive Detail?";
	bool appEdit = true;
> = false;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Texture", useNormalMap )
#endif

texture snowDiffuse : TEXTURE
<
	string UIName = "Snow Diffuse";
	bool appEdit = true;
>;

texture snowSpecular : TEXTURE
<
	string UIName = "Snow Specular";
	bool appEdit = true;
>;

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( snowNormal, "Snow Normal", useNormalMap )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float snowMinSpec
<
	string UIName = "Snow Min Specular";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float snowMaxSpec
<
	string UIName = "Snow Max Specular";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float snowGlobalSpecular
<
	string UIName = "Snow Global Specular";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;



//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D broadDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="broadDiffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < broadDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D middleDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="middleDiffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < middleDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D detailDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="detailDiffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < detailDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

sampler2D snowDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="snowDiffuse"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < snowDiffuse >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D snowSpecularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="snowSpecular"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < snowSpecular >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D snowNormalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="snowNormal"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < snowNormal >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position  : POSITION;													// Object space position
#ifdef _3DSMAX_
	float4 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 0 texture coord - N.B. Max requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord1 : TEXCOORD2;													// UV channel 1 texture coord
#else
	float4 colour    : COLOR0;														// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 0 texture coord
	float2 texCoord1 : TEXCOORD1;													// UV channel 1 texture coord
#endif
	float3 normal    : NORMAL;														// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal  : BINORMAL;													// Object space normal
#endif
};


// Output structure (also input to the fragment shader)
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD5;														// Vertex colour
	float3 texCoord0PlusSnowFactor		: TEXCOORD0;				// UV coords for texture channel 0 + snow factor in z
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float2 texCoord2	: TEXCOORD2;												// UV coords for texture channel 2
	float3 normal			: TEXCOORD3;												// Normal vector (world space)
	float4 eye				: TEXCOORD4;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD6;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD7;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 8 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 6 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//
// TODO: rearrange to transform eye and lights into local space before
// vector calculations.

VSOUTPUT Ground_SnowcoverVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord0PlusSnowFactor.xy = _input.texCoord0;
	_output.texCoord1 = _input.texCoord1;

	// Calculate scaled and rotated version of texture channel 1
	float s, c;
	sincos( radians( detailUVRotation ), s, c );
#ifdef _3DSMAX_
	float2 texCoord2 = float2( _input.texCoord1.x, -_input.texCoord1.y );
#else
	float2 texCoord2 = float2( _input.texCoord1.x, _input.texCoord1.y );
#endif
	texCoord2 -= 0.5f;
	texCoord2 = float2( ( texCoord2.x * c ) + ( texCoord2.y * -s ),
											( texCoord2.x * s ) + ( texCoord2.y * c ) ) * float2( detailUScale, detailVScale ) + 0.5f;
	texCoord2.y *= -1.0f;
	_output.texCoord2 = texCoord2;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate vert's world-space normal
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate snow factor (rubbish but clear)
#ifdef _3DSMAX_
	float upwardNormalComponent = normal.z;
#else
	float upwardNormalComponent = normal.y;
#endif

	float snowFactor;
	if ( upwardNormalComponent < snowThreshold )
	{
		snowFactor = 0.0f;
	}
	else
	{
		if ( upwardNormalComponent < snowThreshold + snowFade )
		{
			snowFactor = ( upwardNormalComponent - snowThreshold ) / max( snowFade, 0.01f );
		}
		else
		{
			snowFactor = 1.0f;
		}
	}
	_output.texCoord0PlusSnowFactor.z = snowFactor;

	// Calculate world-space coordinate frame
	_output.normal   = normal;
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float3 texCoord0PlusSnowFactor	: TEXCOORD0;					// UV coords for texture channel 0 + snow factor in z
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float2 texCoord2	: TEXCOORD2;												// UV coords for texture channel 2
	float3 normal			: TEXCOORD3;												// Normal vector (world space)
	float4 eye				: TEXCOORD4;												// Eye vector (world space)
	float3 tangent		: TEXCOORD6;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD7;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 8 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float3 texCoord0PlusSnowFactor		: TEXCOORD0;				// UV coords for texture channel 0 + snow factor in z
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float2 texCoord2	: TEXCOORD2;												// UV coords for texture channel 2
	float3 normal			: TEXCOORD3_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD4_centroid;								// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD6_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD7_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 8 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT Ground_SnowcoverFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 broadDiffuseTexColour = tex2D( broadDiffuseMap, _input.texCoord0PlusSnowFactor.xy );
	float4 middleDiffuseTexColour = tex2D( middleDiffuseMap, _input.texCoord1 );
	float4 detailDiffuseTexColour = tex2D( detailDiffuseMap, _input.texCoord2 );
	float4 baseSpecularTexColour = tex2D( specularMap, _input.texCoord2 );
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;
	float globalSpecularFactorValue = globalSpecularFactor;

	float4 snowDiffuseColour = tex2D( snowDiffuseMap, _input.texCoord0PlusSnowFactor.xy );
	float4 snowSpecularColour = tex2D( snowSpecularMap, _input.texCoord0PlusSnowFactor.xy );

	// Calculate base colour (combined from broad and detail diffuse)
	float4 diffuseTexColour;
	if ( additiveDetail )
	{
		diffuseTexColour = ( ( broadDiffuseTexColour * middleDiffuseTexColour ) + detailDiffuseTexColour ) * 0.5f;
	}
	else
	{
		diffuseTexColour = broadDiffuseTexColour * middleDiffuseTexColour * detailDiffuseTexColour;
	}

	// Calculate interpolated textures
	diffuseTexColour = lerp( diffuseTexColour, snowDiffuseColour, _input.texCoord0PlusSnowFactor.z );
	float4 specularTexColour = lerp( baseSpecularTexColour, snowSpecularColour, _input.texCoord0PlusSnowFactor.z );

	// Calculate various interpolated factors
	float minSpec = lerp( minSpecPower, snowMinSpec, _input.texCoord0PlusSnowFactor.z );
	float maxSpec = lerp( maxSpecPower, snowMaxSpec, _input.texCoord0PlusSnowFactor.z );
	float globalSpec = lerp( globalSpecularFactor, snowGlobalSpecular, _input.texCoord0PlusSnowFactor.z );

	float4 accumulatedColour = diffuseTexColour * _input.colour;

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Normalise the input tangent and binormal vectors
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 baseNormalMapColour = tex2D( normalMap, _input.texCoord2 );
		float4 snowNormalMapColour = tex2D( snowNormalMap, _input.texCoord0PlusSnowFactor.xy );
		float4 normalMapColour = lerp( baseNormalMapColour, snowNormalMapColour, _input.texCoord0PlusSnowFactor.z );
		float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = broadDiffuseTexColour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}




//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float3 texCoord0PlusSnowFactor		: TEXCOORD0;				// UV coords for texture channel 0 + snow factor in z
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float2 texCoord2	: TEXCOORD2;												// UV coords for texture channel 2
};



VSOUTPUT_LD Ground_SnowcoverLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord0PlusSnowFactor.xy = _input.texCoord0;
	_output.texCoord1 = _input.texCoord1;

	// Calculate scaled and rotated version of texture channel 1
	float s, c;
	sincos( radians( detailUVRotation ), s, c );
#ifdef _3DSMAX_
	float2 texCoord2 = float2( _input.texCoord1.x, -_input.texCoord1.y );
#else
	float2 texCoord2 = float2( _input.texCoord1.x, _input.texCoord1.y );
#endif
	texCoord2 -= 0.5f;
	texCoord2 = float2( ( texCoord2.x * c ) + ( texCoord2.y * -s ),
											( texCoord2.x * s ) + ( texCoord2.y * c ) ) * float2( detailUScale, detailVScale ) + 0.5f;
	texCoord2.y *= -1.0f;
	_output.texCoord2 = texCoord2;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world-space normal
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;
	
	// Calculate snow factor (rubbish but clear)
#ifdef _3DSMAX_
	float upwardNormalComponent = normal.z;
#else
	float upwardNormalComponent = normal.y;
#endif

	float snowFactor;
	if ( upwardNormalComponent < snowThreshold )
	{
		snowFactor = 0.0f;
	}
	else
	{
		if ( upwardNormalComponent < snowThreshold + snowFade )
		{
			snowFactor = ( upwardNormalComponent - snowThreshold ) / max( snowFade, 0.01f );
		}
		else
		{
			snowFactor = 1.0f;
		}
	}
	_output.texCoord0PlusSnowFactor.z = snowFactor;

	_output.colour = _input.colour;
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

PSOUTPUT Ground_SnowcoverLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 broadDiffuseTexColour = tex2D( broadDiffuseMap, _input.texCoord0PlusSnowFactor.xy );
	float4 middleDiffuseTexColour = tex2D( middleDiffuseMap, _input.texCoord1 );
	float4 detailDiffuseTexColour = tex2D( detailDiffuseMap, _input.texCoord2 );

	float4 snowDiffuseColour = tex2D( snowDiffuseMap, _input.texCoord0PlusSnowFactor.xy );

	// Calculate base colour (combined from broad and detail diffuse)
	float4 diffuseTexColour;
	if ( additiveDetail )
	{
		diffuseTexColour = ( ( broadDiffuseTexColour * middleDiffuseTexColour ) + detailDiffuseTexColour ) * 0.5f;
	}
	else
	{
		diffuseTexColour = broadDiffuseTexColour * middleDiffuseTexColour * detailDiffuseTexColour;
	}

	// Calculate interpolated textures
	diffuseTexColour = lerp( diffuseTexColour, snowDiffuseColour, _input.texCoord0PlusSnowFactor.z );


	float4 accumulatedColour = diffuseTexColour * _input.colour;
	accumulatedColour.w = broadDiffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}




//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Ground_Snowcover
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground_Snowcover";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Ground_Snowcover_LowDetail";
	int    lowDetailDeferredID		= 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx Ground_SnowcoverVertexShader();
		PixelShader = compile sce_fp_rsx Ground_SnowcoverFragmentShader();
#else		
		VertexShader = compile vs_3_0 Ground_SnowcoverVertexShader();
		PixelShader = compile ps_3_0 Ground_SnowcoverFragmentShader();
#endif		
	}
}


technique Ground_Snowcover_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Ground_Snowcover_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx Ground_SnowcoverLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx Ground_SnowcoverLowDetailFragmentShader();
#else		
		VertexShader = compile vs_3_0 Ground_SnowcoverLowDetailVertexShader();
		PixelShader = compile ps_3_0 Ground_SnowcoverLowDetailFragmentShader();
#endif		
	}
}
